import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.image.BufferedImage;
import java.awt.image.DataBufferByte;
import java.io.*;
import java.util.Arrays;
import java.util.*;
import java.text.*;
import java.nio.ByteBuffer;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.w3c.dom.Document;
import org.w3c.dom.*;


import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.DocumentBuilder;
import org.xml.sax.SAXException;
import org.xml.sax.SAXParseException;

public class DataCreator
{
    public static boolean debug = true;
    public static boolean training = false;
    public static String dataFolder = "data/";
    public int seed = 0;

    public void printMessage(String s) {
        if (debug) {
            System.out.println(s);
        }
    }

    public class Coordinate
    {
        public double latitude;
        public double longitude;

        public Coordinate(double lat, double lon) {
            latitude = lat;
            longitude = lon;
        }
    }

    public class Quake
    {
        public int timeSecs;
        public double depth;
        public double magnitude;
        public Coordinate coord;
        public Quake(int tim, Coordinate loc, double dep, double mag) {
            coord = loc;
            depth = dep;
            magnitude = mag;
            timeSecs = tim;
        }
    }

    public class DataSet
    {
        Coordinate[] sites;
        Quake[] quakes;
        int sampleRate;
        int numOfSites;
        int numOfEMA;
        int numOfQuakes;

        int[] rawData = null;
        byte[] result = null;
        double[] EMA = null;

        String gtfStartTime, gtfEQtime;
        double gtfMagnitude, gtfLatitude, gtfLongitude, gtfDistToEQ, gtfEQSec;
        int gtfEQHour, gtfSite;

        DocumentBuilderFactory docBuilderFactory = null;
        DocumentBuilder docBuilder = null;

        public void loadSiteInfo(String sXmlFile) throws Exception
        {
            // load site information
            Document doc = docBuilder.parse (new File(sXmlFile));
            doc.getDocumentElement ().normalize ();

            NodeList listOfSites = doc.getElementsByTagName("Site");
            numOfSites = listOfSites.getLength();
            printMessage("Total number of sites: " + numOfSites);

            sites = new Coordinate[numOfSites];
            for (int s=0;s<numOfSites;s++) {
                Element siteElement = (Element)listOfSites.item(s);
                if (s==0) {
                    sampleRate = Integer.parseInt( siteElement.getAttribute("sample_rate") );
                    printMessage("Sample Rate = "+sampleRate);
                }
                double lat = Double.parseDouble( siteElement.getAttribute("latitude") );
                double lon = Double.parseDouble( siteElement.getAttribute("longitude") );
                sites[s] = new Coordinate(lat, lon);
                printMessage("site "+s+": "+lat+","+lon);
            }
            // allocate memory for hourly data
            rawData = new int[numOfSites*3*3600*sampleRate];
            result = new byte[3600*sampleRate*4];
        }

        public void loadEarthMagneticActivity(String sXmlFile) throws Exception
        {
            // load earth magnetic activity
            Document doc = docBuilder.parse (new File(sXmlFile));
            doc.getDocumentElement ().normalize ();

            NodeList listOfEMA = doc.getElementsByTagName("kp_hr");
            numOfEMA = listOfEMA.getLength();
            printMessage("Total number of EM activities: " + numOfEMA);

            EMA = new double[numOfEMA];
            for (int i=0;i<numOfEMA;i++) {
                EMA[i] = Double.parseDouble(listOfEMA.item(i).getFirstChild().getNodeValue());
            }
        }

        public void loadOtherQuakes(String sXmlFile) throws Exception
        {
            // load earth magnetic activity
            Document doc = docBuilder.parse (new File(sXmlFile));
            doc.getDocumentElement ().normalize ();

            NodeList listOfQuakes = doc.getElementsByTagName("Quake");
            numOfQuakes = listOfQuakes.getLength();
            printMessage("Total number of other quakes: " + numOfQuakes);

            quakes = new Quake[numOfQuakes];
            for (int i=0;i<numOfQuakes;i++) {
                Element quakeElement = (Element)listOfQuakes.item(i);
                int secs = Integer.parseInt( quakeElement.getAttribute("secs") );
                double lat = Double.parseDouble( quakeElement.getAttribute("latitude") );
                double lon = Double.parseDouble( quakeElement.getAttribute("longitude") );
                double depth = Double.parseDouble( quakeElement.getAttribute("depth") );
                double mag = Double.parseDouble( quakeElement.getAttribute("magnitude") );
                quakes[i] = new Quake(secs, new Coordinate(lat, lon), depth, mag);
            }
        }

        public double[] getOtherQuakes(int hour)
        {
            int hStart = hour*3600;
            int hEnd = (hour+1)*3600;
            int numInHour = 0;
            for (int i=0;i<numOfQuakes;i++) {
                if (quakes[i].timeSecs>=hStart && quakes[i].timeSecs<hEnd) numInHour++;
            }
            double[] oQuake = new double[numInHour*5];
            int q = 0;
            for (int i=0;i<numOfQuakes;i++) {
                if (quakes[i].timeSecs>=hStart && quakes[i].timeSecs<hEnd) {
                    oQuake[q] = quakes[i].coord.latitude;
                    oQuake[q+1] = quakes[i].coord.longitude;
                    oQuake[q+2] = quakes[i].depth;
                    oQuake[q+3] = quakes[i].magnitude;
                    oQuake[q+4] = quakes[i].timeSecs;
                    q+=5;
                }
            }
            return oQuake;
        }

        public DataSet(String sFolder) throws Exception
        {
            docBuilderFactory = DocumentBuilderFactory.newInstance();
            docBuilder = docBuilderFactory.newDocumentBuilder();
            loadSiteInfo(sFolder+"SiteInfo.xml");
            loadEarthMagneticActivity(sFolder+"Kp.xml");
            loadOtherQuakes(sFolder+"Quakes.xml");
        }

        public void readGTF() throws Exception
        {
            BufferedReader br = new BufferedReader(new FileReader("gtf.csv"));
            int numOfCases = Integer.parseInt(br.readLine());
            for (int i=0;i<numOfCases;i++)
            {
                String s = br.readLine();
                String[] token = s.split(",");
                int setID = Integer.parseInt(token[0]);
                if (setID==seed)
                {
                    gtfStartTime = token[1];
                    gtfEQtime = token[2];
                    gtfMagnitude = Double.parseDouble(token[3]);
                    gtfLatitude = Double.parseDouble(token[4]);
                    gtfLongitude = Double.parseDouble(token[5]);
                    gtfSite = Integer.parseInt(token[6]);
                    gtfDistToEQ = Double.parseDouble(token[7]);
                    // Calculate number of hours till EQ
                    SimpleDateFormat ft = new SimpleDateFormat("yyyy-MM-dd hh:mm:ss");
                    Date date1 = ft.parse(gtfStartTime);
                    long startMSec = date1.getTime();
                    Date date2 = ft.parse(gtfEQtime);
                    long eqMSec = date2.getTime();
                    gtfEQSec = (eqMSec-startMSec)/1000;
                    gtfEQHour = (int)(gtfEQSec / (60*60));
                    printMessage("Quake happened at hour = " + gtfEQHour +" and second = " + gtfEQSec+ " at row "+(gtfEQSec-3600.0*gtfEQHour)*sampleRate);
                    break;
                }
            }
            br.close();
        }

        public int[] loadHour(String sFolder, int h) throws Exception
        {
            String fname = sFolder + "test" + seed + "_" + h + ".bin";
            printMessage("loading "+fname);
            int[] dt = new int[numOfSites*3*3600*sampleRate];
            File file = new File(fname);
            InputStream input = new BufferedInputStream(new FileInputStream(file));
            DataInputStream din = new DataInputStream(input);
            int prev = -1;
            int diff;
            for (int i=0;i<dt.length;i++)
            {
                int b1 = (int)din.readByte(); if (b1<0) b1+=256;
                if ((b1&3)==1) {
                    diff = (b1>>2)-(1<<5);
                    dt[i] = prev + diff;
                } else if ((b1&3)==2) {
                    int b2 = (int)din.readByte(); if (b2<0) b2+=256;
                    diff = ((b1+(b2<<8))>>2)-(1<<13);
                    dt[i] = prev + diff;
                } else if ((b1&3)==3) {
                    int b2 = (int)din.readByte(); if (b2<0) b2+=256;
                    int b3 = (int)din.readByte(); if (b3<0) b3+=256;
                    diff = ((b1+(b2<<8)+(b3<<16))>>2)-(1<<21);
                    dt[i] = prev + diff;
                } else
                {
                    int b2 = (int)din.readByte(); if (b2<0) b2+=256;
                    int b3 = (int)din.readByte(); if (b3<0) b3+=256;
                    int b4 = (int)din.readByte(); if (b4<0) b4+=256;
                    dt[i] = ((b1+(b2<<8)+(b3<<16)+(b4<<24))>>2)-1;
                }
                prev = dt[i];
            }
            input.close();
            return dt;
        }

    }

    public double doExec() throws Exception {
        try {
            DataSet dset = new DataSet(dataFolder+seed+"/");
            // load ground truth data
            dset.readGTF();

            BufferedWriter writer = new BufferedWriter(new FileWriter(dataFolder+"data"+seed+".csv"));
            final int fftSample = 1<<14;
            final int mid = (1<<24) / 2;
            final int hourRange = 24*21;    //3 weeks

            double[][][] real = new double[dset.sites.length][3][fftSample];
            double[][][] prevReal = new double[dset.sites.length][3][fftSample/2];
            int sampleOffset = 0;
            for (int h=Math.max(0, dset.gtfEQHour - 2 * hourRange);h<dset.gtfEQHour;h++) {  //start at the hour 6 weeks before quake.
                int[] hourlyData = dset.loadHour(dataFolder + seed + "/", h);
                int numSamples = hourlyData.length / dset.sites.length / 3;
                for (int j = 0; j < dset.sites.length; j++) {
                    
                    //quake occurs within 3 weeks -> 2
                    //quake occurs within 6 weeks -> 1
                    //quake does not occur at this site ->0
                    //in another program ("training.py"), these will be modified
                    int occur = j == dset.gtfSite ? (h < dset.gtfEQHour - hourRange ? 1 : 2) : 0;

                    double[] prev = h==dset.gtfEQHour - hourRange ? new double[]{mid, mid, mid} : new double[]{ prevReal[j][0][prevReal.length-1], prevReal[j][1][prevReal.length-1], prevReal[j][2][prevReal.length-1] };
                    for (int sample = 0; sample < numSamples; sample++) {
                        int i = (sample + sampleOffset) % (fftSample / 2);
                        for (int c = 0; c < 3; c++) {
                            int v = hourlyData[j * 3 * numSamples + c * numSamples + sample];
                            if (v < 0) v = (int)prev[c];
                            real[j][c][fftSample / 2 + i] = v;
                            prev[c] = v;
                        }
                        if (i == (fftSample / 2) - 1) {
                            boolean firstFlg = h == dset.gtfEQHour - hourRange && sample < fftSample / 2;

                            double tStart = 3600 * h + (double) (sample - fftSample + 1) / dset.sampleRate;
                            double tEnd = 3600 * h + (double) sample / dset.sampleRate;
                            int day = (int) Math.floor(tStart / 3600 / 24);
                            int hour = (day % (3600 * 24)) / 3600;

                            final double[] hz = new double[]{ 0, 0.004, 0.005, 0.006, 0.008, 0.01, 0.02, 0.05, 0.1, 0.2, 0.5, 1, 2, 5};
                            double[][] features = new double[3][hz.length];
                            for (int c = 0; c < 3; c++) {
                                double[] imag = new double[fftSample];
                                for (int ii = 0; ii < fftSample / 2; ii++) {
                                    real[j][c][ii] = prevReal[j][c][ii];
                                    prevReal[j][c][ii] = real[j][c][(fftSample / 2) + ii];
                                }

                                Fft.transform(real[j][c], imag);

                                double sum = 0;
                                int ihz = 0;
                                for (int ii = 0; ii < fftSample / 2; ii++) {
                                    double a = Math.sqrt(real[j][c][ii] * real[j][c][ii] + imag[ii] * imag[ii]);
                                    sum += a;
                                    if (ihz < hz.length && dset.sampleRate * ii >= fftSample * hz[ihz]) {
                                        features[c][ihz] = sum;
                                        sum = 0;
                                        ihz++;
                                    }
                                }
                            }
                            if (!firstFlg) {
                                writer.write(occur + "");
                                for (int c = 0; c < 3; c++) {
                                    for (int ii = 0; ii < features[c].length; ii++) writer.write("," + features[c][ii]);
                                }
                                for (int c = 0; c < 3; c++) {
                                    for (int ii = 0; ii < features[c].length; ii++)
                                        writer.write("," + (features[c][ii] / (features[(c + 1) % 3][ii] + 1)));
                                }
                                writer.write("," + dset.EMA[hour]);
                                writer.newLine();
                            }
                        }
                    }
                }
                sampleOffset = (sampleOffset + numSamples) % (fftSample / 2);
            }
            writer.flush();
            writer.close();
        } catch (Exception e) {
            System.out.println("FAILURE: " + e.getMessage());
            e.printStackTrace();
            throw e;
        }
        return -1.0;
    }

    public static void main(String[] args) throws Exception {
            ExecutorService executor = Executors.newFixedThreadPool(16);
            for (int seed = 1; seed <= 300; seed++) {
                final int _seed = seed;
                executor.execute(new Runnable() {
                    public void run() {
                        try {
                            System.out.println(_seed);
                            DataCreator creator = new DataCreator();
                            creator.seed = _seed;
                            creator.doExec();
                        } catch (Exception e) {
                            System.out.println("FAILURE: " + e.getMessage());
                            e.printStackTrace();
                        }
                    }
                });
            }
            executor.shutdown();
            executor.awaitTermination(72, TimeUnit.HOURS);
    }

    class ErrorStreamRedirector extends Thread {
        public BufferedReader reader;

        public ErrorStreamRedirector(InputStream is) {
            reader = new BufferedReader(new InputStreamReader(is));
        }

        public void run() {
            while (true) {
                String s;
                try {
                    s = reader.readLine();
                } catch (Exception e) {
                    // e.printStackTrace();
                    return;
                }
                if (s == null) {
                    break;
                }
                System.out.println(s);
            }
        }
    }
}


/*
 * Free FFT and convolution (Java)
 *
 * Copyright (c) 2014 Project Nayuki
 * http://www.nayuki.io/page/free-small-fft-in-multiple-languages
 *
 * (MIT License)
 * Permission is hereby granted, free of charge, to any person obtaining a copy of
 * this software and associated documentation files (the "Software"), to deal in
 * the Software without restriction, including without limitation the rights to
 * use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of
 * the Software, and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 * - The above copyright notice and this permission notice shall be included in
 *   all copies or substantial portions of the Software.
 * - The Software is provided "as is", without warranty of any kind, express or
 *   implied, including but not limited to the warranties of merchantability,
 *   fitness for a particular purpose and noninfringement. In no event shall the
 *   authors or copyright holders be liable for any claim, damages or other
 *   liability, whether in an action of contract, tort or otherwise, arising from,
 *   out of or in connection with the Software or the use or other dealings in the
 *   Software.
 */


class Fft {
    
    /*
     * Computes the discrete Fourier transform (DFT) of the given complex vector, storing the result back into the vector.
     * The vector can have any length. This is a wrapper function.
     */
    public static void transform(double[] real, double[] imag) {
        if (real.length != imag.length)
            throw new IllegalArgumentException("Mismatched lengths");
        
        int n = real.length;
        if (n == 0)
            return;
        else if ((n & (n - 1)) == 0)  // Is power of 2
            transformRadix2(real, imag);
        else  // More complicated algorithm for arbitrary sizes
            transformBluestein(real, imag);
    }
    
    
    /*
     * Computes the inverse discrete Fourier transform (IDFT) of the given complex vector, storing the result back into the vector.
     * The vector can have any length. This is a wrapper function. This transform does not perform scaling, so the inverse is not a true inverse.
     */
    public static void inverseTransform(double[] real, double[] imag) {
        transform(imag, real);
    }
    
    
    /*
     * Computes the discrete Fourier transform (DFT) of the given complex vector, storing the result back into the vector.
     * The vector's length must be a power of 2. Uses the Cooley-Tukey decimation-in-time radix-2 algorithm.
     */
    public static void transformRadix2(double[] real, double[] imag) {
        // Initialization
        if (real.length != imag.length)
            throw new IllegalArgumentException("Mismatched lengths");
        int n = real.length;
        int levels = 31 - Integer.numberOfLeadingZeros(n);  // Equal to floor(log2(n))
        if (1 << levels != n)
            throw new IllegalArgumentException("Length is not a power of 2");
        double[] cosTable = new double[n / 2];
        double[] sinTable = new double[n / 2];
        for (int i = 0; i < n / 2; i++) {
            cosTable[i] = Math.cos(2 * Math.PI * i / n);
            sinTable[i] = Math.sin(2 * Math.PI * i / n);
        }
        
        // Bit-reversed addressing permutation
        for (int i = 0; i < n; i++) {
            int j = Integer.reverse(i) >>> (32 - levels);
            if (j > i) {
                double temp = real[i];
                real[i] = real[j];
                real[j] = temp;
                temp = imag[i];
                imag[i] = imag[j];
                imag[j] = temp;
            }
        }
        
        // Cooley-Tukey decimation-in-time radix-2 FFT
        for (int size = 2; size <= n; size *= 2) {
            int halfsize = size / 2;
            int tablestep = n / size;
            for (int i = 0; i < n; i += size) {
                for (int j = i, k = 0; j < i + halfsize; j++, k += tablestep) {
                    double tpre =  real[j+halfsize] * cosTable[k] + imag[j+halfsize] * sinTable[k];
                    double tpim = -real[j+halfsize] * sinTable[k] + imag[j+halfsize] * cosTable[k];
                    real[j + halfsize] = real[j] - tpre;
                    imag[j + halfsize] = imag[j] - tpim;
                    real[j] += tpre;
                    imag[j] += tpim;
                }
            }
            if (size == n)  // Prevent overflow in 'size *= 2'
                break;
        }
    }
    
    
    /*
     * Computes the discrete Fourier transform (DFT) of the given complex vector, storing the result back into the vector.
     * The vector can have any length. This requires the convolution function, which in turn requires the radix-2 FFT function.
     * Uses Bluestein's chirp z-transform algorithm.
     */
    public static void transformBluestein(double[] real, double[] imag) {
        // Find a power-of-2 convolution length m such that m >= n * 2 + 1
        if (real.length != imag.length)
            throw new IllegalArgumentException("Mismatched lengths");
        int n = real.length;
        if (n >= 0x20000000)
            throw new IllegalArgumentException("Array too large");
        int m = Integer.highestOneBit(n * 2 + 1) << 1;
        
        // Trignometric tables
        double[] cosTable = new double[n];
        double[] sinTable = new double[n];
        for (int i = 0; i < n; i++) {
            int j = (int)((long)i * i % (n * 2));  // This is more accurate than j = i * i
            cosTable[i] = Math.cos(Math.PI * j / n);
            sinTable[i] = Math.sin(Math.PI * j / n);
        }
        
        // Temporary vectors and preprocessing
        double[] areal = new double[m];
        double[] aimag = new double[m];
        for (int i = 0; i < n; i++) {
            areal[i] =  real[i] * cosTable[i] + imag[i] * sinTable[i];
            aimag[i] = -real[i] * sinTable[i] + imag[i] * cosTable[i];
        }
        double[] breal = new double[m];
        double[] bimag = new double[m];
        breal[0] = cosTable[0];
        bimag[0] = sinTable[0];
        for (int i = 1; i < n; i++) {
            breal[i] = breal[m - i] = cosTable[i];
            bimag[i] = bimag[m - i] = sinTable[i];
        }
        
        // Convolution
        double[] creal = new double[m];
        double[] cimag = new double[m];
        convolve(areal, aimag, breal, bimag, creal, cimag);
        
        // Postprocessing
        for (int i = 0; i < n; i++) {
            real[i] =  creal[i] * cosTable[i] + cimag[i] * sinTable[i];
            imag[i] = -creal[i] * sinTable[i] + cimag[i] * cosTable[i];
        }
    }
    
    
    /*
     * Computes the circular convolution of the given real vectors. Each vector's length must be the same.
     */
    public static void convolve(double[] x, double[] y, double[] out) {
        if (x.length != y.length || x.length != out.length)
            throw new IllegalArgumentException("Mismatched lengths");
        int n = x.length;
        convolve(x, new double[n], y, new double[n], out, new double[n]);
    }
    
    
    /*
     * Computes the circular convolution of the given complex vectors. Each vector's length must be the same.
     */
    public static void convolve(double[] xreal, double[] ximag, double[] yreal, double[] yimag, double[] outreal, double[] outimag) {
        if (xreal.length != ximag.length || xreal.length != yreal.length || yreal.length != yimag.length || xreal.length != outreal.length || outreal.length != outimag.length)
            throw new IllegalArgumentException("Mismatched lengths");
        
        int n = xreal.length;
        xreal = xreal.clone();
        ximag = ximag.clone();
        yreal = yreal.clone();
        yimag = yimag.clone();
        
        transform(xreal, ximag);
        transform(yreal, yimag);
        for (int i = 0; i < n; i++) {
            double temp = xreal[i] * yreal[i] - ximag[i] * yimag[i];
            ximag[i] = ximag[i] * yreal[i] + xreal[i] * yimag[i];
            xreal[i] = temp;
        }
        inverseTransform(xreal, ximag);
        for (int i = 0; i < n; i++) {  // Scaling (because this FFT implementation omits it)
            outreal[i] = xreal[i] / n;
            outimag[i] = ximag[i] / n;
        }
    }
    
}
